# CB User API Component for Joomla 5

A simple REST API component for Joomla 5 that retrieves Community Builder user profile data. This component allows you to fetch CB user fields from one Joomla site to another via a secure API endpoint.

## Features

- Retrieve Community Builder profile data via REST API
- Token-based authentication
- Returns user data in JSON format
- Includes core Joomla user fields (id, email)
- Includes Community Builder fields (firstname, lastname, avatar, address, city, state, country)
- Automatic avatar URL conversion
- Built with modern Joomla 5 namespace structure

## Installation

1. Download or create the component files in this structure:
```
com_cbuserapi/
├── cbuserapi.xml
├── site/
│   └── src/
│       └── Controller/
│           └── DisplayController.php
└── admin/
    ├── services/
    │   └── provider.php
    └── src/
        └── Extension/
            └── CbuserApiComponent.php
```

2. Zip the `com_cbuserapi` folder
3. Install via Joomla Administrator → System → Extensions → Install
4. The component will be installed and ready to use

## Configuration

### Set Your Authentication Token

**IMPORTANT:** Before using the API, you must set a secure authentication token.

1. Open `site/src/Controller/DisplayController.php`
2. Find this line in the `validateToken()` method:
```php
$validToken = 'YOUR_SECRET_API_TOKEN_HERE'; // Change this!
```
3. Replace `YOUR_SECRET_API_TOKEN_HERE` with a strong, random token
4. Example: `$validToken = 'k9mP2xL8vQ4wN7rT3jH6yU1zF5bS0aD9';`

**Tip:** Generate a secure token using:
- Online tool: https://randomkeygen.com/
- Command line: `openssl rand -hex 32`
- PHP: `bin2hex(random_bytes(32))`

### Customize Namespace (Optional)

If you want to use your own namespace instead of `YourName`:

1. Edit all PHP files and replace `YourName` with your namespace (e.g., `MyCompany`)
2. Update the namespace in `cbuserapi.xml`:
```xml
<namespace path="src">MyCompany\Component\Cbuserapi</namespace>
```

## Usage

### API Endpoint

```
GET https://yoursite.com/index.php?option=com_cbuserapi&task=getUser&user_id={USER_ID}&token={YOUR_TOKEN}
```

### Parameters

- `option` - Component name (always `com_cbuserapi`)
- `task` - Action to perform (currently only `getUser`)
- `user_id` - The Joomla user ID to retrieve (integer)
- `token` - Your authentication token (string)

### Example Request

```
https://yoursite.com/index.php?option=com_cbuserapi&task=getUser&user_id=123&token=k9mP2xL8vQ4wN7rT3jH6yU1zF5bS0aD9
```

### Response Format

**Success (200):**
```json
{
  "success": true,
  "data": {
    "id": 123,
    "email": "john.doe@example.com",
    "firstname": "John",
    "lastname": "Doe",
    "avatar": "tn123_avatar.jpg",
    "avatar_url": "https://yoursite.com/images/comprofiler/tn123_avatar.jpg",
    "cb_address": "123 Main Street",
    "cb_city": "Dallas",
    "cb_state": "TX",
    "cb_country": "United States"
  }
}
```

**Invalid Token (401):**
```json
{
  "success": false,
  "message": "Invalid or missing authentication token"
}
```

**Invalid User ID (400):**
```json
{
  "success": false,
  "message": "Invalid user ID"
}
```

**User Not Found (404):**
```json
{
  "success": false,
  "message": "User not found"
}
```

**Server Error (500):**
```json
{
  "success": false,
  "message": "Database error: [error details]"
}
```

## Calling from Another Joomla Site

### Using PHP cURL

```php
<?php
$userId = 123;
$token = 'k9mP2xL8vQ4wN7rT3jH6yU1zF5bS0aD9';
$apiUrl = 'https://yoursite.com/index.php?option=com_cbuserapi&task=getUser&user_id=' . $userId . '&token=' . $token;

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $apiUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($httpCode === 200) {
    $data = json_decode($response, true);
    if ($data['success']) {
        $user = $data['data'];
        echo "Name: " . $user['firstname'] . " " . $user['lastname'];
        echo "Email: " . $user['email'];
        // Use other fields as needed
    }
}
?>
```

### Using JavaScript/AJAX

```javascript
const userId = 123;
const token = 'k9mP2xL8vQ4wN7rT3jH6yU1zF5bS0aD9';
const apiUrl = `https://yoursite.com/index.php?option=com_cbuserapi&task=getUser&user_id=${userId}&token=${token}`;

fetch(apiUrl)
  .then(response => response.json())
  .then(data => {
    if (data.success) {
      console.log('User:', data.data);
      console.log('Name:', data.data.firstname, data.data.lastname);
      console.log('Email:', data.data.email);
    } else {
      console.error('Error:', data.message);
    }
  })
  .catch(error => console.error('Request failed:', error));
```

## Security Best Practices

### Required Security Measures

1. **Always Use HTTPS**
   - Never call the API over HTTP
   - Ensure your SSL certificate is valid
   - Tokens transmitted over HTTP can be intercepted

2. **Strong Token**
   - Use a random, cryptographically secure token
   - Minimum 32 characters
   - Store securely on both sites

3. **Keep Token Secret**
   - Never commit tokens to version control
   - Don't expose in client-side JavaScript
   - Store in configuration files outside web root if possible

### Optional Enhanced Security

**IP Whitelisting:**

Add this to the `validateToken()` method in `DisplayController.php`:

```php
private function validateToken($token)
{
    // IP whitelist
    $allowedIPs = ['192.168.1.100', '203.0.113.45'];
    $clientIP = $_SERVER['REMOTE_ADDR'];
    
    if (!in_array($clientIP, $allowedIPs)) {
        return false;
    }
    
    // Token validation
    $validToken = 'YOUR_SECRET_API_TOKEN_HERE';
    return $token === $validToken;
}
```

**Database Token Storage:**

Instead of hardcoding the token, store it in a database table:

1. Create a table `#__api_tokens`:
```sql
CREATE TABLE `#__api_tokens` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `token` varchar(255) NOT NULL,
  `description` varchar(255) DEFAULT NULL,
  `enabled` tinyint(1) DEFAULT 1,
  `created` datetime DEFAULT NULL,
  PRIMARY KEY (`id`),
  UNIQUE KEY `token` (`token`)
);
```

2. Replace the `validateToken()` method:
```php
private function validateToken($token)
{
    $db = Factory::getDbo();
    $query = $db->getQuery(true)
        ->select('COUNT(*)')
        ->from($db->quoteName('#__api_tokens'))
        ->where($db->quoteName('token') . ' = ' . $db->quote($token))
        ->where($db->quoteName('enabled') . ' = 1');
    
    $db->setQuery($query);
    return (int)$db->loadResult() > 0;
}
```

**Rate Limiting:**

Add basic rate limiting to prevent abuse:

```php
private function checkRateLimit($token)
{
    $db = Factory::getDbo();
    $now = time();
    $window = 60; // 1 minute
    $maxRequests = 60; // 60 requests per minute
    
    // Check recent requests
    $query = $db->getQuery(true)
        ->select('COUNT(*)')
        ->from($db->quoteName('#__api_requests'))
        ->where($db->quoteName('token') . ' = ' . $db->quote($token))
        ->where($db->quoteName('timestamp') . ' > ' . ($now - $window));
    
    $db->setQuery($query);
    $count = (int)$db->loadResult();
    
    if ($count >= $maxRequests) {
        return false;
    }
    
    // Log this request
    $query = $db->getQuery(true)
        ->insert($db->quoteName('#__api_requests'))
        ->columns(['token', 'timestamp'])
        ->values($db->quote($token) . ', ' . $now);
    
    $db->setQuery($query);
    $db->execute();
    
    return true;
}
```

## Extending the Component

### Adding More CB Fields

To retrieve additional Community Builder fields:

1. Open `site/src/Controller/DisplayController.php`
2. Add fields to the SELECT query:
```php
->select([
    'u.id',
    'u.email',
    'cb.firstname',
    'cb.lastname',
    'cb.avatar',
    'cb.cb_address',
    'cb.cb_city',
    'cb.cb_state',
    'cb.cb_country',
    'cb.cb_phone',        // Add new field
    'cb.cb_company'       // Add new field
])
```

### Creating Additional Endpoints

To add new functionality (e.g., update user data):

1. Create a new method in `DisplayController.php`:
```php
public function updateUser()
{
    // Your update logic here
}
```

2. Call it with:
```
https://yoursite.com/index.php?option=com_cbuserapi&task=updateUser&user_id=123&token=YOUR_TOKEN
```

## Troubleshooting

### "Invalid or missing authentication token"
- Verify the token in the URL matches the one in `DisplayController.php`
- Check for extra spaces or special characters in the token

### "User not found"
- Verify the user ID exists in the `#__users` table
- Check that the user has a Community Builder profile

### "Database error"
- Verify Community Builder is installed
- Check that CB tables exist (`#__comprofiler`)
- Verify CB field names match your installation

### Component won't install
- Check that all files are in the correct folder structure
- Verify XML syntax in `cbuserapi.xml`
- Check Joomla error logs for details

### Empty or null field values
- Some CB fields may be optional and empty
- Check the CB field names match your installation
- Some fields might use different column names

## Requirements

- Joomla 5.x
- Community Builder installed and configured
- PHP 7.4 or higher
- MySQL 5.6 or higher

## Support

For issues or questions:
- Check the Troubleshooting section above
- Review Joomla error logs
- Verify Community Builder is properly installed

## License

GNU General Public License version 2 or later

## Changelog

### Version 1.0.0 (2025-01)
- Initial release
- Basic user data retrieval
- Token authentication
- JSON response format

## Credits

Created for cross-site Joomla integration with Community Builder user data.
