<?php
/**
 * @package     CB User API
 * @subpackage  com_cbuserapi
 * @copyright   Copyright (C) 2025. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Kyle\Component\Cbuserapi\Site\Controller;

defined('_JEXEC') or die;

use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;

/**
 * CB User API Controller
 */
class DisplayController extends BaseController
{
    /**
     * Default view
     */
    protected $default_view = 'api';
    
    /**
     * Get user data by user ID
     */
    public function getUser()
    {
        // Set JSON response header
        header('Content-Type: application/json');
        
        // Get input
        $app = Factory::getApplication();
        $input = $app->input;
        
        // Get parameters
        $userId = $input->getInt('user_id', 0);
        $token = $input->getString('token', '');
        
        // Validate token
        if (!$this->validateToken($token)) {
            http_response_code(401);
            echo json_encode([
                'success' => false,
                'message' => 'Invalid or missing authentication token'
            ]);
            $app->close();
            return;
        }
        
        // Validate user ID
        if ($userId <= 0) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => 'Invalid user ID'
            ]);
            $app->close();
            return;
        }
        
        // Get database
        $db = Factory::getDbo();
        
        // Build query
        $query = $db->getQuery(true)
            ->select([
                'u.id',
                'u.email',
                'cb.firstname',
                'cb.lastname',
                'cb.avatar',
                'cb.cb_address',
                'cb.cb_city',
                'cb.cb_state',
                'cb.cb_country'
            ])
            ->from($db->quoteName('#__users', 'u'))
            ->leftJoin($db->quoteName('#__comprofiler', 'cb') . ' ON u.id = cb.user_id')
            ->where($db->quoteName('u.id') . ' = ' . $db->quote($userId));
        
        try {
            $db->setQuery($query);
            $result = $db->loadObject();
            
            if ($result) {
                // Process avatar to full URL if exists
                if (!empty($result->avatar)) {
                    $result->avatar_url = $this->getAvatarUrl($result->avatar);
                }
                
                echo json_encode([
                    'success' => true,
                    'data' => $result
                ]);
            } else {
                http_response_code(404);
                echo json_encode([
                    'success' => false,
                    'message' => 'User not found'
                ]);
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'Database error: ' . $e->getMessage()
            ]);
        }
        
        $app->close();
    }
    
    /**
     * Validate authentication token
     * 
     * @param string $token The token to validate
     * @return bool True if valid
     */
    private function validateToken($token)
    {
        // IMPORTANT: Replace this with your actual token validation
        // Options:
        // 1. Check against a database table of valid API tokens
        // 2. Use Joomla's user token
        // 3. Implement JWT validation
        // 4. Use a shared secret between sites
        
        // Simple example - check against a configured token
        $validToken = 'YOUR_SECRET_API_TOKEN_HERE'; // Change this!
        
        return $token === $validToken;
        
        // Better approach - check database:
        /*
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('COUNT(*)')
            ->from($db->quoteName('#__api_tokens'))
            ->where($db->quoteName('token') . ' = ' . $db->quote($token))
            ->where($db->quoteName('enabled') . ' = 1');
        $db->setQuery($query);
        return (int)$db->loadResult() > 0;
        */
    }
    
    /**
     * Convert CB avatar filename to full URL
     * 
     * @param string $avatar Avatar filename
     * @return string Full URL to avatar
     */
    private function getAvatarUrl($avatar)
    {
        if (empty($avatar)) {
            return '';
        }
        
        // CB typically stores avatars in /images/comprofiler/
        $baseUrl = rtrim(Uri::root(), '/');
        return $baseUrl . '/images/comprofiler/' . $avatar;
    }
}
