/**
 * AI Button for Joomla Editors (TinyMCE and JCE) - Joomla 5 Compatible
 */

import { JoomlaEditorButton } from 'editor-api';

// Get AJAX URL from Joomla options
const options = Joomla.getOptions('plg_editors-xtd_aibutton') || {};
const AJAX_URL = options.ajaxUrl || '';

// Create modal HTML
const createModal = () => {
    const modal = document.createElement('div');
    modal.id = 'ai-modal';
    modal.innerHTML = `
        <div class="ai-modal-overlay">
            <div class="ai-modal-content">
                <div class="ai-modal-header">
                    <h3>AI Assistant</h3>
                    <button class="ai-close-btn" type="button">&times;</button>
                </div>
                <div class="ai-modal-body">
                    <textarea id="ai-prompt" placeholder="Enter your prompt here..." rows="4"></textarea>
                    <div class="ai-actions">
                        <button id="ai-insert-btn" type="button" class="btn btn-primary">Insert</button>
                        <button id="ai-replace-btn" type="button" class="btn btn-secondary">Replace Selection</button>
                        <button id="ai-append-btn" type="button" class="btn btn-secondary">Append</button>
                    </div>
                    <div id="ai-result" style="display:none;">
                        <h4>AI Response:</h4>
                        <div id="ai-response"></div>
                    </div>
                    <div id="ai-loading" style="display:none;">
                        <div class="spinner"></div>
                        <p>Generating response...</p>
                    </div>
                </div>
            </div>
        </div>
    `;
    document.body.appendChild(modal);
    return modal;
};

// Get the active editor instance
const getActiveEditor = () => {
    // Try Joomla 5 Editor API first
    if (typeof Joomla !== 'undefined' && Joomla.Editor) {
        const activeEditor = Joomla.Editor.getActive();
        if (activeEditor) {
            return {
                type: 'joomla5',
                instance: activeEditor
            };
        }
    }

    // Try TinyMCE
    if (typeof tinymce !== 'undefined' && tinymce.activeEditor) {
        return {
            type: 'tinymce',
            instance: tinymce.activeEditor,
            id: tinymce.activeEditor.id
        };
    }

    // Try JCE - it's based on TinyMCE so check tinymce object
    if (typeof tinymce !== 'undefined' && tinymce.editors && tinymce.editors.length > 0) {
        const editor = tinymce.editors[0];
        return {
            type: 'jce',
            instance: editor,
            id: editor.id
        };
    }

    // Fallback: try to find any TinyMCE editor
    if (typeof tinymce !== 'undefined') {
        for (let id in tinymce.editors) {
            return {
                type: 'jce',
                instance: tinymce.editors[id],
                id: id
            };
        }
    }

    return null;
};

// Insert content into editor
const insertContent = (content, mode = 'insert') => {
    const editor = getActiveEditor();
    if (!editor) {
        alert('No editor found');
        return;
    }

    // Convert plain text line breaks to HTML
    const htmlContent = content.replace(/\n\n/g, '</p><p>').replace(/\n/g, '<br>');
    const formattedContent = '<p>' + htmlContent + '</p>';

    if (editor.type === 'joomla5') {
        // Joomla 5 Editor API
        const ed = editor.instance;
        switch(mode) {
            case 'replace':
                ed.replaceSelection(formattedContent);
                break;
            case 'append':
                const currentContent = ed.getValue();
                ed.setValue(currentContent + formattedContent);
                break;
            case 'insert':
            default:
                ed.replaceSelection(formattedContent);
                break;
        }
    } else if (editor.type === 'tinymce') {
        const ed = editor.instance;
        const editorId = editor.id;
        
        // Check if editor is in source/code mode (works for both TinyMCE and JCE)
        const textarea = document.getElementById(editorId);
        const isSourceMode = textarea && textarea.style.display !== 'none' && textarea.offsetParent !== null;
        
        // In code mode, insert directly into the textarea
        if (isSourceMode && textarea) {
            const start = textarea.selectionStart || 0;
            const end = textarea.selectionEnd || 0;
            const currentValue = textarea.value;
            
            switch(mode) {
                case 'replace':
                    textarea.value = currentValue.substring(0, start) + content + currentValue.substring(end);
                    textarea.selectionStart = textarea.selectionEnd = start + content.length;
                    break;
                case 'append':
                    textarea.value = currentValue + content;
                    textarea.selectionStart = textarea.selectionEnd = textarea.value.length;
                    break;
                case 'insert':
                default:
                    textarea.value = currentValue.substring(0, start) + content + currentValue.substring(end);
                    textarea.selectionStart = textarea.selectionEnd = start + content.length;
                    break;
            }
            textarea.focus();
            // Trigger change event so editor knows content changed
            const event = new Event('change', { bubbles: true });
            textarea.dispatchEvent(event);
            return;
        }
        
        // Visual mode - use normal editor methods
        switch(mode) {
            case 'replace':
                if (ed.selection && ed.selection.setContent) {
                    ed.selection.setContent(formattedContent);
                }
                break;
            case 'append':
                const currentContent = ed.getContent();
                ed.setContent(currentContent + formattedContent);
                break;
            case 'insert':
            default:
                if (ed.insertContent) {
                    ed.insertContent(formattedContent);
                } else if (ed.execCommand) {
                    ed.execCommand('mceInsertContent', false, formattedContent);
                }
                break;
        }
    } else if (editor.type === 'jce') {
        const ed = editor.instance;
        const editorId = editor.id;
        
        // Check if editor is in source/code mode by checking if the textarea is visible
        const textarea = document.getElementById(editorId);
        const isSourceMode = textarea && textarea.style.display !== 'none' && textarea.offsetParent !== null;
        
        // In code mode, insert directly into the textarea
        if (isSourceMode && textarea) {
            const start = textarea.selectionStart;
            const end = textarea.selectionEnd;
            const currentValue = textarea.value;
            
            switch(mode) {
                case 'replace':
                    textarea.value = currentValue.substring(0, start) + content + currentValue.substring(end);
                    textarea.selectionStart = textarea.selectionEnd = start + content.length;
                    break;
                case 'append':
                    textarea.value = currentValue + content;
                    textarea.selectionStart = textarea.selectionEnd = textarea.value.length;
                    break;
                case 'insert':
                default:
                    textarea.value = currentValue.substring(0, start) + content + currentValue.substring(end);
                    textarea.selectionStart = textarea.selectionEnd = start + content.length;
                    break;
            }
            textarea.focus();
            // Trigger change event so JCE knows content changed
            const event = new Event('change', { bubbles: true });
            textarea.dispatchEvent(event);
            return;
        }
        
        // Visual mode - use normal editor methods
        switch(mode) {
            case 'replace':
                if (ed.selection && ed.selection.setContent) {
                    ed.selection.setContent(formattedContent);
                } else {
                    ed.execCommand('mceReplaceContent', false, formattedContent);
                }
                break;
            case 'append':
                const currentContent = ed.getContent();
                ed.setContent(currentContent + formattedContent);
                break;
            case 'insert':
            default:
                if (ed.execCommand) {
                    ed.execCommand('mceInsertContent', false, formattedContent);
                } else if (ed.insertContent) {
                    ed.insertContent(formattedContent);
                }
                break;
        }
    }
};

// Call Claude API via Joomla AJAX proxy
const callClaudeAPI = async (prompt) => {
    if (!AJAX_URL) {
        throw new Error('AJAX URL not configured');
    }

    const formData = new FormData();
    formData.append('prompt', prompt);
    formData.append(Joomla.getOptions('csrf.token'), '1');

    const response = await fetch(AJAX_URL, {
        method: 'POST',
        body: formData
    });

    if (!response.ok) {
        throw new Error('Network error');
    }

    const data = await response.json();
    
    if (!data.success) {
        throw new Error(data.message || 'API request failed');
    }

    return data.data.text;
};

// Clean up markdown code fences from response
const cleanResponse = (text) => {
    // Remove markdown code fences like ```html, ```javascript, etc.
    return text.replace(/^```[\w]*\n?/gm, '').replace(/\n?```$/gm, '').trim();
};

// Show modal
const showModal = (editor, options, button) => {
    let modal = document.getElementById('ai-modal');
    if (!modal) {
        modal = createModal();
        
        // Set up event listeners
        modal.querySelector('.ai-close-btn').addEventListener('click', () => {
            modal.style.display = 'none';
        });

        modal.querySelector('.ai-modal-overlay').addEventListener('click', (e) => {
            if (e.target.classList.contains('ai-modal-overlay')) {
                modal.style.display = 'none';
            }
        });

        const handleAction = async (mode) => {
            const prompt = document.getElementById('ai-prompt').value.trim();
            if (!prompt) {
                alert('Please enter a prompt');
                return;
            }

            const loadingDiv = document.getElementById('ai-loading');
            const resultDiv = document.getElementById('ai-result');
            const responseDiv = document.getElementById('ai-response');

            try {
                loadingDiv.style.display = 'block';
                resultDiv.style.display = 'none';

                const response = await callClaudeAPI(prompt);
                const cleanedResponse = cleanResponse(response);
                
                loadingDiv.style.display = 'none';
                responseDiv.textContent = cleanedResponse;
                resultDiv.style.display = 'block';

                // Insert the response into editor
                insertContent(cleanedResponse, mode);

            } catch (error) {
                loadingDiv.style.display = 'none';
                alert('Error: ' + error.message);
            }
        };

        modal.querySelector('#ai-insert-btn').addEventListener('click', () => handleAction('insert'));
        modal.querySelector('#ai-replace-btn').addEventListener('click', () => handleAction('replace'));
        modal.querySelector('#ai-append-btn').addEventListener('click', () => handleAction('append'));
    }

    modal.style.display = 'block';
    document.getElementById('ai-prompt').focus();
};

// Register action with Joomla Editor Button API
JoomlaEditorButton.registerAction('aibutton.open', showModal);
