<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  Editors-xtd.aibutton
 *
 * @copyright   Copyright (C) 2024 Your Name. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace kdd\Plugin\EditorsXtd\AIButton\Extension;

use Joomla\CMS\Editor\Button\Button;
use Joomla\CMS\Event\Editor\EditorButtonsSetupEvent;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Event\SubscriberInterface;
use Joomla\CMS\Response\JsonResponse;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * AI Button Plugin
 *
 * @since  1.0.0
 */
final class AIButton extends CMSPlugin implements SubscriberInterface
{
    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return  array
     *
     * @since   1.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onEditorButtonsSetup' => 'onEditorButtonsSetup',
            'onAjaxAibutton' => 'onAjaxAibutton',
        ];
    }

    /**
     * Display the button
     *
     * @param   EditorButtonsSetupEvent  $event  The event
     *
     * @return  void
     *
     * @since   1.0.0
     */
    public function onEditorButtonsSetup(EditorButtonsSetupEvent $event): void
    {
        $subject = $event->getButtonsRegistry();
        $disabled = $event->getDisabledButtons();

        if (\in_array('aibutton', $disabled)) {
            return;
        }

        // Register the JavaScript file
        $this->getApplication()->getDocument()->getWebAssetManager()
            ->registerAndUseScript(
                'plg_editors-xtd_aibutton',
                'plg_editors-xtd_aibutton/aibutton.js',
                [],
                ['type' => 'module'],
                ['editors']
            )
            ->registerAndUseStyle(
                'plg_editors-xtd_aibutton',
                'plg_editors-xtd_aibutton/aibutton.css'
            );

        // Pass API URL to JavaScript (not the API key for security)
        $this->getApplication()->getDocument()->addScriptOptions('plg_editors-xtd_aibutton', [
            'ajaxUrl' => \Joomla\CMS\Uri\Uri::root() . 'index.php?option=com_ajax&plugin=aibutton&group=editors-xtd&format=json'
        ]);

        // Create the button
        $button = new Button(
            'aibutton',
            [
                'text' => 'AI Assistant',
                'icon' => 'brain',
                'action' => 'aibutton.open',
                'onclick' => ''
            ]
        );

        $subject->add($button);
    }

    /**
     * AJAX handler to proxy Claude API requests
     *
     * @return  void
     *
     * @since   1.0.0
     */
    public function onAjaxAibutton(): void
    {
        // Check for valid session
        $app = $this->getApplication();
        
        if (!$app->getSession()->checkToken('get') && !$app->getSession()->checkToken('post')) {
            echo new JsonResponse(['error' => 'Invalid token'], 'Invalid token', true);
            $app->close();
        }

        // Get the API key from plugin params
        $apiKey = $this->params->get('api_key', '');
        
        if (empty($apiKey)) {
            echo new JsonResponse(['error' => 'API key not configured'], 'API key not configured', true);
            $app->close();
        }

        // Get the prompt from the request
        $input = $app->input;
        $prompt = $input->post->getString('prompt', '');
        
        if (empty($prompt)) {
            echo new JsonResponse(['error' => 'No prompt provided'], 'No prompt provided', true);
            $app->close();
        }

        // Make the API request to Claude
        $data = [
            'model' => 'claude-sonnet-4-20250514',
            'max_tokens' => 4096,
            'messages' => [
                [
                    'role' => 'user',
                    'content' => $prompt
                ]
            ]
        ];

        $ch = curl_init('https://api.anthropic.com/v1/messages');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'x-api-key: ' . $apiKey,
            'anthropic-version: 2023-06-01'
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            echo new JsonResponse(['error' => $error], 'cURL error', true);
            $app->close();
        }

        if ($httpCode !== 200) {
            $errorData = json_decode($response, true);
            $errorMessage = $errorData['error']['message'] ?? 'API request failed';
            echo new JsonResponse(['error' => $errorMessage], $errorMessage, true);
            $app->close();
        }

        $responseData = json_decode($response, true);
        $text = $responseData['content'][0]['text'] ?? '';

        echo new JsonResponse(['text' => $text]);
        $app->close();
    }
}
